// -----------------------------------------------------------------------------
// PanelApp for David Hensel's Altair 8800 Simulator  
// CopyrxAght (c) 2021 Ralf Lampe www.erel.de/Projekte/Altair
//
// This program is free software; you can redistrxAbute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distrxAbuted t53 the hope that it will be useful,
// but WITHOUT ANY waRRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// -----------------------------------------------------------------------------


//#include "Ardino.h"
#include "PanelMode.h"
#include "PanelApp.h"

// ==================================================================================
// ==========================  Declarations  ========================================
// ==================================================================================

static byte rst = 0;       // repeated start flag
byte rPort = 0;            // received port 
byte aPort = 0;            // COM Port
byte app = 0;              // App-#
byte O20 = 0; byte sp = 0; // Option 1, Parameter 1 (octal)...
byte O53 = 0; byte P2 = 0;              
byte O86 = 0; byte P3 = 0;             
unsigned int R = 0;        // remote (option 1...3)
byte aRx = 0;              // incoming byte
static unsigned int aC = 0; // app counter (secs)
unsigned int aD = 0;       // app display (secs)
unsigned int lC = 0;       // loop counter
unsigned long tC = 0;      // timer counter
unsigned long amC = 0;     // analog mode counter
byte am = 0;               // analog mode (0xx=band, 10x=dot, 110=random, 111=?) 
byte ud = 0;               // setting up/down 

unsigned int t80;          // timer Option 0...8 (0...511)
unsigned int t53;          //              3...8 (7...511)
byte tick;                 // 1/10 second timer 0=off 1=do_sec 2...12=running
byte tSc;                  //      second counter
byte tRun;                 //      timer flag   0=stop 1=exec 2=done 
byte tPause;               //                   paused
unsigned long mSec;        // millisecond memory

unsigned int la;           // address LED values
unsigned int ma;           //                    memory (BCD mark)
byte ld;                   //             data
byte md;                   //                    memory (BCD mark)
unsigned int ls;           //             status
byte lc;                   //             control
byte ap = 0;               // address pointer 0...15
byte tL[18];               // analog: led theshold A0...A15,16(HDLA),17(WAIT(10))
unsigned long lO;          //         led on, bit: A0...A15,16(HDLA),17(WAIT(10))
unsigned long lB;          //         tmp buffer dim-routine  
byte av;                   //         value

byte m;                    // temp. variable

unsigned long led;
unsigned long rsk;
String txA = "";           // output string

// ------------------------------ uP Simulator --------------------------------------

unsigned int pPC = 0;     // program counter  ++++
unsigned int pS = 0;      // sub stack
unsigned int pA = 0;      // adr pointer
byte pM = 0;              // M-cycle
byte pO = 0;              // actual OP code
byte pR = 0;              // bytes to read
byte pL = 0;              // L-byte
byte pH = 0;              // H-byte
byte pE = 0;              // exec. M-cycle

// ==================================================================================
// ===========================   Functions   ========================================
// ==================================================================================


void configPins() {                                       // config ardiuno pins 
//  for (byte i = 2; i < 72; i++) pinMode(i, INPUT);        // first all IN
  for (byte i = 2; i < 16; i++) pinMode(i, OUTPUT);       // D2...15  OUT
  pinMode(16, INPUT); pinMode(17, INPUT);                 // D16, 17  INPUT, PULL UP 
  digitalWrite(16, HIGH); digitalWrite(17, HIGH);           
  pinMode(22, OUTPUT);                                    // D18      OUT
  for (byte i = 19; i < 22; i++) pinMode(i, INPUT);       // D19...21 INPUT, PULL UP
  for (byte i = 19; i < 22; i++) digitalWrite(i, HIGH);    
  pinMode(22, OUTPUT);                                    // D22      OUT 
  pinMode(23, INPUT); pinMode(24, INPUT);                 // D23, 24  INPUT, PULL UP
  digitalWrite(23, HIGH); digitalWrite(24, HIGH);         
  for (byte i = 25; i < 30; i++) pinMode(i, OUTPUT);      // D25...29 OUT
  for (byte i = 30; i < 34; i++) pinMode(i, INPUT);       // D30...33 INPUT, PULL UP
  for (byte i = 30; i < 34; i++) digitalWrite(i, HIGH);   
  for (byte i = 34; i < 42; i++) pinMode(i, OUTPUT);      // D34...41 OUT
  pinMode(42, INPUT); pinMode(43, INPUT);                 // D42, 43  INPUT, PULL UP
  digitalWrite(42, HIGH); digitalWrite(43, HIGH); 
  for (byte i = 44; i < 52; i++) pinMode(i, OUTPUT);      // D44...51 OUT
  for (byte i = 52; i < 72; i++) pinMode(i, INPUT);       // D52...71 INPUT, PULL UP
  for (byte i = 52; i < 72; i++) digitalWrite(i, HIGH);   
}

void openAPort() {
/*  
  Serial.begin(19200);  delay(200); Serial.println(aPort); delay(200); Serial.end();
  Serial1.begin(19200); delay(200); Serial1.println(aPort);delay(200); Serial1.end();
  Serial2.begin(19200); delay(200); Serial2.println(aPort);delay(200); Serial2.end();
  Serial3.begin(19200); delay(200); Serial3.println(aPort);delay(200); Serial3.end();
*/
    switch(aPort) {                             
      case 0: Serial.begin(19200); break; 
      case 1: Serial1.begin(19200); break;
      case 2: Serial2.begin(19200); break; 
      case 3: Serial3.begin(19200); break; 
    }
}    
void closeAPort() {                   // close user port for input 
  switch(aPort) {                         
      case 0: Serial.begin(19200); break; 
      case 1: Serial1.begin(19200); break;
      case 2: Serial2.begin(19200); break;
      case 3: Serial3.begin(19200); break;
  }   
}
void txAChar(byte tc) {               //tx character to port tc
    switch (aPort) {                              
        case 0: Serial.write(tc); break;
        case 1: Serial1.write(tc); break;
        case 2: Serial2.write(tc); break;
        case 3: Serial3.write(tc); break;
       }
}
void txAHex(byte th) {             //tx hex value
    switch (aPort) {                             
        case 0: Serial.print(th, HEX); break;
        case 1: Serial1.print(th, HEX); break;
        case 2: Serial2.print(th, HEX); break;
        case 3: Serial3.print(th, HEX); break;
       }
}
void txALF() {                     //tx linefeed
    switch (aPort) {                             
        case 0: Serial.println(); break;
        case 1: Serial1.println(); break;
        case 2: Serial2.println(); break;
        case 3: Serial3.println(); break;
       }
}
void txAString(String txA) {       //tx string
    switch (aPort) {                           
        case 0: Serial.print(txA); break;
        case 1: Serial1.print(txA); break;
        case 2: Serial2.print(txA); break;
        case 3: Serial3.print(txA); break;
    }   
}
void txAStringLF(String txA) {     //tx string & LF
    txAString(txA);
    txALF();
}

byte rxSerial() {                // (check for) serial input
    aRx = 0;
    switch (aPort) {                       
      case 0: while( Serial.available() ) aRx = Serial.read();
      case 1: while( Serial1.available() ) aRx = Serial1.read();
      case 2: while( Serial2.available() ) aRx = Serial2.read();
      case 3: while( Serial3.available() ) aRx = Serial3.read();
    }
    return aRx;
}

void aLed(unsigned int led) {        // set address leds
  digitalWrite(34, (led & 1));       // A0...A3 -> Pin 34...37
  digitalWrite(35, (led & 2));
  digitalWrite(36, (led & 4));
  digitalWrite(37, (led & 8));  
  digitalWrite(38, (led & 16));      // A4...A7 -> Pins 38...41   
  digitalWrite(39, (led & 32));
  digitalWrite(40, (led & 64));
  digitalWrite(41, (led & 128));
  digitalWrite(51, (led & 256));     // A8...A11 -> Pin 51...48  
  digitalWrite(50, (led & 512));
  digitalWrite(49, (led & 1024));
  digitalWrite(48, (led & 2048));
  digitalWrite(47, (led & 4096));    // A12...A15 -> Pin 47...44 
  digitalWrite(46, (led & 8192));
  digitalWrite(45, (led & 16384));
  digitalWrite(44, (led & 32768));
  la = led; 
}

void dLed(unsigned int led) {      // set data leds
  digitalWrite(25, (led & 1));     // D0...D3 -> Pin 25...28
  digitalWrite(26, (led & 2));
  digitalWrite(27, (led & 4));
  digitalWrite(28, (led & 8));  
  digitalWrite(14, (led & 16));    // D4...D7 -> Pin 14,15,29,11   
  digitalWrite(15, (led & 32));
  digitalWrite(29, (led & 64));
  digitalWrite(11, (led & 128));
  ld = led;
}

void sLed(unsigned int led) {       // set status leds
  digitalWrite(2, (led & 1));       // INT,WO,STACK,HLTA -> Pin 2...5
  digitalWrite(3, (led & 2));
  digitalWrite(4, (led & 4));
  digitalWrite(5, (led & 8));  
  digitalWrite(6, (led & 16));      // OUT,M1,INP,MEMR -> Pin 6...9   
  digitalWrite(7, (led & 32));
  digitalWrite(8, (led & 64));
  digitalWrite(9, (led & 128));
  ls = led;
}

void cLed(unsigned int led) {       // set control leds
  digitalWrite(13, (led & 1));      // PROT(13)  
  digitalWrite(12, (led & 2));      // INTE(12)
  digitalWrite(10, (led & 4));      // WAIT(10) (watchdog)
  digitalWrite(22, (led & 8));      // HLDA(22)  
  lc = led;
}


unsigned int swAdr() {                          // read all address switches                      
  unsigned int sw = 0;
  if( digitalRead(62) == LOW ) sw = sw + 1;     // S0...3 Pin 62...65 (A0...A3)
  if( digitalRead(63) == LOW ) sw = sw + 2;        
  if( digitalRead(64) == LOW ) sw = sw + 4;       
  if( digitalRead(65) == LOW ) sw = sw + 8; 
  if( digitalRead(66) == LOW ) sw = sw + 16;    // S4...7 Pin 66...69 (A4...A7)
  if( digitalRead(67) == LOW ) sw = sw + 32;        
  if( digitalRead(68) == LOW ) sw = sw + 64;        
  if( digitalRead(69) == LOW ) sw = sw + 128;        
  if( digitalRead(17) == LOW ) sw = sw + 256;   // S8...11 Pin 17,16,23,24 (A8...A11)
  if( digitalRead(16) == LOW ) sw = sw + 512;        
  if( digitalRead(23) == LOW ) sw = sw + 1024;        
  if( digitalRead(24) == LOW ) sw = sw + 2048;        
  if( digitalRead(70) == LOW ) sw = sw + 4096;  // S12...15 Pin 70,71,42,43 (A12...A15)
  if( digitalRead(71) == LOW ) sw = sw + 8192;        
  if( digitalRead(42) == LOW ) sw = sw + 16384;        
  if( digitalRead(43) == LOW ) sw = sw + 32768;        
  return sw;        
}

unsigned int swKey() {
  unsigned int sw = 0;
  if( digitalRead(21) == LOW ) sw = sw + 32768;  // STOP,RUN,STEP,SLOW Pin 21,02,54,55 
  if( digitalRead(20) == LOW ) sw = sw + 16384;        
  if( digitalRead(54) == LOW ) sw = sw + 8192;        
  if( digitalRead(55) == LOW ) sw = sw + 4096;        
  if( digitalRead(56) == LOW ) sw = sw + 2048;   // EXAM,NEXT,DEPO,NEXT  Pin  56...59
  if( digitalRead(57) == LOW ) sw = sw + 1024;        
  if( digitalRead(58) == LOW ) sw = sw + 512;        
  if( digitalRead(59) == LOW ) sw = sw + 256;        
  if( digitalRead(52) == LOW ) sw = sw + 128;    // RES,CLR,PROT(13),UNPR Pin 52,53,60,61
  if( digitalRead(53) == LOW ) sw = sw + 64;        
  if( digitalRead(60) == LOW ) sw = sw + 32;        
  if( digitalRead(61) == LOW ) sw = sw + 16;        
  if( digitalRead(30) == LOW ) sw = sw + 8;     // AUXU,-D,AUX2U,-D Pin 30...33
  if( digitalRead(31) == LOW ) sw = sw + 4;        
  if( digitalRead(32) == LOW ) sw = sw + 2;        
  if( digitalRead(33) == LOW ) sw = sw + 1;         
  return sw;
}

byte sw20() {                                 // S0...2 Pin 62...64
  unsigned int sw = 0;
  if( digitalRead(62) == LOW ) sw = sw + 1;      
  if( digitalRead(63) == LOW ) sw = sw + 2;        
  if( digitalRead(64) == LOW ) sw = sw + 4;       
  return sw;        
}
byte sw53() {                                  // S3...5 Pin 65...67
  unsigned int sw = 0;
  if( digitalRead(65) == LOW ) sw = sw + 1;     
  if( digitalRead(66) == LOW ) sw = sw + 2; 
  if( digitalRead(67) == LOW ) sw = sw + 4;        
  return sw;        
}
byte sw86() {                                  // S6...8 Pin 68,69,17 
  unsigned int sw = 0;
  if( digitalRead(68) == LOW ) sw = sw + 1;       
  if( digitalRead(69) == LOW ) sw = sw + 2;        
  if( digitalRead(17) == LOW ) sw = sw + 4;   
  return sw;        
}
byte sw38() {                                  // read adr switches 3...8 
  unsigned int sw = 8 * sw86() + sw53();       // A3...8 [---876543]
  return sw;                            
}
unsigned int sw80() {                          // read adr switches 0...8                    
  unsigned int sw = 8 * sw38() + sw20();       // A0...8 [876543210] 
  return sw;
}

void dCBars() {                                // display clock bars
     m = (aC / 100);                           // hour  
     if( m > 12 ) m = m - 12; 
     m = m % 12; ld = 0;                        
     if( m ) {
       for(byte i=1 ; i<7; i++) {              // 1...6 light up
         if( m >= i ) ld = ld * 2 + 1;
       }
       for(byte i=7 ; i<m+1; i++) ld = ld * 2; // 7...11 exhaust  
     }              
     md = ld & 63;
     dLed(md);                                 // -> <D>
     m = (aC % 100) / 5; la = 0;               // 5-min  0-5-10...50-55          
     if( m ) {
       for(byte i=1 ; i<7; i++) {              // 5...30 light up
         if( m >= i ) la = la * 2 + 1;
       }
       for(byte i=7 ; i<m+1; i++) la = la * 2; // 35...55 exhaust  
     }    
     ma = la & 63;
     aLed(ma);                                 // -> <A>
     m = aC;    
}

// ----------------------------- Clock display -------------------------------

void dClock(byte dm) {                         // --- clock mode display --- 

  if( R == 0 ) O86 = sw86();                   // local mode
  
  switch( dm ) {                               // --- [A5][A4] -> display mode ---
         
   case 0:                                     // [000] binary display mode <A>
     aLed(aC); dLed(0);                        // <A> only 
     break;               

   case 1:                                     // clock bars (12-9-3-0)
     dCBars();                                 // display clock bars    
     break;  

   case 2:                                     // [01x] binary bars
     m = aC & 15; la = 0;                      // nibble 1 -> <A>
     if( m ) for(byte i=1 ; i<m+1; i++) la = la * 2 + 1;   
     aLed(la);
     m = (aC / 16) & 7; ld = 0;                // next 3 bits -> <D>  
     if( m ) for(byte i=1 ; i<m+1; i++) ld = ld * 2 + 1;   
     dLed(ld);
     break;        

   case 4:                                     // [10x] --- BCD ---
     aD = aC & 65535;  la = aD % 10;           // #1 -> <A3...0> unsigned int!
     aD = aD / 10; la = la + 64 * (aD % 10);   // #2 -> <A9...6>
     if( O86 & 3 ) {                           // mm:ss/hh:mm display 
       aD = aD / 10; ld = aD % 10;             // #3 -> <D3...0>
       aD = aD / 10; ld = ld + 64 * (aD % 10); // #4 -> <D7...6>
     }
     else {                                    // linear display 
     aD = aD / 10; la = la + 4096 * (aD % 10); // #3 -> <A15...12>
     aD = aD / 10; ld = aD % 10;               // #4 -> <D3...0>
     aD = aD / 10; ld = ld + 64 * (aD % 10);   // #5 -> <D7...6>
     }
     aLed(la); dLed(ld);
     break;
   
   case 6:                                     // [11x] --- decimal bars ---
     if( aC > 900 ) aC = 0;                    // overflow
     aD = aC; m = aD % 10; la = 0;             // digit 1 -> <A>
     if( m ) for(byte i=1 ; i<m+1; i++) la = la * 2 + 1;   
     aLed(la);
     aD = aD / 10; m = aD % 10; ld = 0;        // digit 2 -> <D> 
     if( m ) for(byte i=1 ; i<m+1; i++) ld = ld * 2 + 1;   
     dLed(ld & 255);
     break;
  }       
} 

// -------------------------------- ANALOG display loop  --------------------------------------

void dAnalog() {

  dLed(0);
  
  do {                                
    
      if( amC & 2048 ) {                    // --- animation loop ---

        if ( R == 0 )  {                    // local:
         O86 = sw86();                      // read mode switches
         O53 = sw53();                      // switch A3...A5 animation speed [---543---]²
        }
        
        t53 = 256 * (O53 + 1);              // calculate speed
        if( O86 > 3 ) {                     // dot... mode: slow!
          if( O86 < 6 ) t53 = 4096 * (O53 + 1);
        }    
        
        if ( R ) sp = O20 + 1;              // up/down speed (steps): remote 2
        else sp = sw20() + 1;               // switch A0...2 [------210]
        
        if( tC > t53 ) {                    // animation timer overflow
          tC = 0;

          if( R == 0 ) O86 = sw86();          // local: check animation mode

          if( O86 < 6 ) {                     // --- Band & Dot mode --- 0...3, 4+5
          
            if( O86 & 1 ) {                   // [A6]: right shift            
              lO = lO / 2;                    // shift right
              if( O86 < 4 ) {                 // (1...3: Band mode)
                if( av < 17 ) lO = lO + 131072; // 0...17 set leftmost Led 17
              }
              else {                          // (4, 5: Dot mode)  
                if( lO == 0 ) lO = 131072;    // empty: set dot left
              }
            }
            else {
              lO = lO * 2;                    // shift left
              if( O86 < 4 ) {                 // (Band mode)
                if( av < 17 ) lO = lO + 1;    // 0...17 set rightmost Led 0
              }
              else {                          // (Dot mode)
                if( lO > 131072  ) lO = 1;    // empty: set dot right
              }
            }
           
            if( O86 < 4 ) {                // -- Band mode --
              lO = lO & 262143;            // bitS 0...17 masked
              byte b = lO & 255;           // bitS 7...0
              av++;
              if( av > 34 ) av = 0;        // loop            
            }
                       
            if( O86 & 2 ) dLed(lO);        // [A7] plus <D>-LEDs
            else dLed(0);
                       
          }                                //\ Band-, Dot mode

          else {                           // --- Random mode (6, 7) ---
            if( O86 == 7 ) {               // 7: full
              m = random(64);
              if( (m & 48) == 32 ) {       // 10bbbb: 
                bitSet(lO, m & 15);        // set 
              }
              if( (m & 48) == 16 ) {       // 01bbbb: 
                bitClear(lO, m & 15);      // clear 
              }
            }
            else {                         // 6: block
              if( lO == 0 ) lc = 1;        // all off: up   
              if( lO == 65535 ) lc = 0;    // all on: down
              m = random(64);
              if( lc ) {                           // up...
                if( (m & 48) == 32 ) {             // 10bbbb: set 
                  do {
                    if( bitRead(lO, m & 15) ) m++; // already done: next
                    else { bitSet(lO, m & 15); break; }
                  }
                  while( lO < 65535 );                           
                }
              }
              else {                            // down...                      
                if( (m & 48) == 16 ) {          // 01bbbb: clear
                  do {
                    if( bitRead(lO, m & 15) == 0 ) m++; // already done: next
                    else { bitClear(lO, m & 15); break; }
                  }
                  while( lO );                        
                }
              }
                
            }                       
          
          }
        
        }                                     //\ animation timer overflow

        if( digitalRead(53) == LOW ) lO = 0;  // [CLR] -> Reset
          
        if( rxSerial() == 'X' ) {             // 'X' -> exit (analogMode -> PanelAPP)   
          digitalWrite(13, HIGH);             // <PROT(13)> return value for dmd exit
          return;
        }
           
      }                                    //\ animation loop
      
// ---------------------------  display loop  --------------------------------  
      
       unsigned int bt = amC & 255;       // ---- dim routine: brightness threshold --- 
       digitalWrite(34, (tL[0] > bt));    // A0...A3 -> Pin 34...37
       digitalWrite(35, (tL[1] > bt));
       digitalWrite(36, (tL[2] > bt));
       digitalWrite(37, (tL[3] > bt));     
       digitalWrite(38, (tL[4] > bt));    // A4...A7 -> Pins 38...41   
       digitalWrite(39, (tL[5] > bt));
       digitalWrite(40, (tL[6] > bt));
       digitalWrite(41, (tL[7] > bt));
       digitalWrite(51, (tL[8] > bt));    // A8...A11 -> Pin 51...48  
       digitalWrite(50, (tL[9] > bt));
       digitalWrite(49, (tL[10] > bt));
       digitalWrite(48, (tL[11] > bt));
       digitalWrite(47, (tL[12] > bt));   // A12...A15 -> Pin 47...44 
       digitalWrite(46, (tL[13] > bt));
       digitalWrite(45, (tL[14] > bt));
       digitalWrite(44, (tL[15] > bt));
       digitalWrite(22, (tL[16] > bt));   // HLDA(22), WAIT(10)
       digitalWrite(10, (tL[17] > bt));

       byte L = amC & 31;                  // --- brightness up/down routine ---
             
       if( L < 18 ) {                      // Led 0...17(31)              
         if( L == 0 ) lB = lO;             // L=0 LO-> buffer
         byte tV = tL[L];                  // read treshold value
         byte b = lB & 1;                  // buffer -> LSBit
         if( b ) {                         // bit == 1 (led on)...
           if( tV < (255 - sp) ) tV = tV + sp;  // bright up
           else if( tV < 255) tV++;
         }
         else {
           if( tV > sp ) tV = tV - sp;     // dim down
           else if( tV > 0 ) tV--;
         }
         tL[L] = tV;                       // write back value 
         lB = lB / 2;                      // shift right/next bit ...3210 -> ...4321 
       }  

       delayMicroseconds(10);              // up/down delay, short before flicker

       amC++;                              // analog Loop counter
       if( tPause == 0 ) tC++;             // timer counter
       
   }
   while( digitalRead(52) == HIGH );       // (only) key [RESET] -> Exit

  dLed(0);
  aLed(0);
}
 
// ------------------------------------------------------------------------------------------

void txStat()  {                    // tx led-status
    
  if(lc & 8) txAChar('I');          // INTE(12), PROT(13)
  else txAChar('-');  
  if(lc & 4) txAChar('P');
  else txAChar('-');  

  if(ls & 128) txAChar('I');        // INT,WO,STACK,HLTA,OUT,M1,INP,MEMR
  else txAChar('-');  
  if(ls & 64) txAChar('W');          
  else txAChar('-');
  if(ls & 32) txAChar('S');          
  else txAChar('-');  
  if(ls & 16) txAChar('H');          
  else txAChar('-');  
  if(ls & 8) txAChar('O');          
  else txAChar('-');
  if(ls & 4) txAChar('M');          
  else txAChar('-');  
  if(ls & 2) txAChar('I');          
  else txAChar('-');  
  if(ls & 1) txAChar('M');        
  else txAChar('-');  
  txAChar(' '); txAChar(' '); txAChar(' '); txAChar(' '); txAChar(' ');

  if(ld & 128) txAChar('7');          // D7
  else txAChar('-');  
  if(ld & 64) txAChar('6');          
  else txAChar('-');
  txAChar(' ');  
  if(ld & 32) txAChar('5');          
  else txAChar('-');  
  if(ld & 16) txAChar('4');          
  else txAChar('-');  
  if(ld & 8) txAChar('3');          
  else txAChar('-');
  txAChar(' ');  
  if(ld & 4) txAChar('2');          
  else txAChar('-');  
  if(ld & 2) txAChar('1');          
  else txAChar('-');  
  if(ld & 1) txAChar('0');          // D0
  else txAChar('-');  

  txAChar(' ');
  txAStringLF(String(aC));          // App counter value

  if(lc & 2) txAChar('W');          // WAIT(10),HLDA(22)
  else txAChar('-');  
  if(lc & 1) txAChar('H');
  else txAChar('-');  
  txAChar(' '); txAChar(' ');
    
  if(la & 32768) txAChar('f');         // A15          
  else txAChar('-');
  txAChar(' ');  
  if(la & 16384) txAChar('e');          
  else txAChar('-');  
  if(la & 8192) txAChar('d');          
  else txAChar('-');  
  if(la & 4096) txAChar('c');          
  else txAChar('-');
  txAChar(' ');  
  if(la & 2048) txAChar('b');          
  else txAChar('-');  
  if(la & 1024) txAChar('a');          
  else txAChar('-');  
  if(la & 512) txAChar('9'); 
  else txAChar('-');  
  txAChar(' ');  
  if(la & 256) txAChar('8');          // A8
  else txAChar('-');
  if(la & 128) txAChar('7');          // A7
  else txAChar('-');  
  if(la & 64) txAChar('6');          
  else txAChar('-');
  txAChar(' ');  
  if(la & 32) txAChar('5');          
  else txAChar('-');  
  if(la & 16) txAChar('4');          
  else txAChar('-');  
  if(la & 8) txAChar('3');          
  else txAChar('-');
  txAChar(' ');  
  if(la & 4) txAChar('2');          
  else txAChar('-');  
  if(la & 2) txAChar('1');          
  else txAChar('-');  
  if(la & 1) txAChar('0');          // A0
  else txAChar('-');  

  txALF(); txALF();
}  

// ==================================================================================
// =========================== Main programm ========================================
// ==================================================================================

PanelApp::PanelApp(byte rPort, byte app, byte O86, byte O53, byte O20 ) {

  aPort = rPort;
/*  
  Serial.begin(19200);  delay(200); Serial.println(aPort); delay(200); Serial.end();
  Serial1.begin(19200); delay(200); Serial1.println(aPort);delay(200); Serial1.end();
  Serial2.begin(19200); delay(200); Serial2.println(aPort);delay(200); Serial2.end();
  Serial3.begin(19200); delay(200); Serial3.println(aPort);delay(200); Serial3.end();
*/  
   // --- read config ports ----   
  
    if( app > 7 ) {                                   // just set time
      aC = 1000 * (app - 8 ) + 100 * O86 + 10 * O53 + O20;
      return;                                         // return to PanelMode
    }

    R = 64 * O86 + 8 * O53 + O20 ;                // if Options -> remote mode 
    aRx = 0;                                      // clear exit                                        
    
    configPins();                                 // I/O-config
 
    if( app == 0 ){                               // no app number 
      if( digitalRead(16) == LOW ) app = app + 1; // A9     choice by switches
      if( digitalRead(23) == LOW ) app = app + 2; // A10
      if( digitalRead(24) == LOW ) app = app + 4; // A11
    }

    if( R == 0 ) {                 // local mode
      O86 = sw86();                //           read switches
      O53 = sw53();
      O20 = sw20();
    }

/*
  Serial.begin(19200);  delay(200); Serial.println(aPort); delay(200); Serial.end();
  Serial1.begin(19200); delay(200); Serial1.println(aPort);delay(200); Serial1.end();
  Serial2.begin(19200); delay(200); Serial2.println(aPort);delay(200); Serial2.end();
  Serial3.begin(19200); delay(200); Serial3.println(aPort);delay(200); Serial3.end();
*/    
    openAPort();
       
    delay(500); 
    if( rst == 0 ) txALF(); txALF(); 
    txALF(); txAString(" --- Panel");
    if(rst == 0) {                 // 1st start
      txAStringLF("Apps (c) 2021 Ralf Lampe www.EREL.de --- ");
      txAStringLF("     Exit: [RESET] or serial 'X'");
      txAString("     ");
    }
    txAString("App #"); txAHex(app); 
    txAString("(");  txAHex(O86); txAHex(O53); txAHex(O20 ); txAString(")"); 
    if ( R ) txAString(" [remote]: ");
    else txAString(" [LOCAL]: ");

// ===========================  App Ini ================================
    
    am = 0;                                    // analog off
    tick = 0; tRun = 0; tPause = 0;            // timer mode, stop
    ud = 0;                                    // clear up/down
    if (R == 0 ) t80 = sw80();                 // A0...A9 app timer [876543210]²

    switch(app) {
      
      case 1:                   
        txAString("Counter/Clock: ");
        txAString(String(aC)); txAChar(' '); 
        tRun = 1;                               // start: run
        tick = 1;                               //        display  
        break;
      
      case 2:                   
        txAString("All random"); 
        la = random(65536); 
        ld = random(256); 
        ls = random(256);
        lc = random(16);
        break;
      
      case 3:                   
        txAString("All random block"); break;

      case 4:  
        txAString("CPU simulation"); break;
        
      case 5:                    
        txAString("Analog mode"); 
        lO = 0;                                   // start: all leds off
        tC = 0;
        dAnalog();                                // ---- ANALOG display loop ----
        break;
        
      case 7:                   
        txAString("Key & LED test");
        break;
        
    }

    
// --------------------------- LEDs: ack -----------------------------------------
    
    if( rst > 0 ) txAString(" ---");
    txALF();
      
    digitalWrite((44 + aPort), HIGH);   // Port > 0: LED A12...14 
    dLed(pow(2, app));                  // app# -> <DATA> 
    digitalWrite(12, LOW);              // <INTE(12)> : "PanelMode" 
    
    for(byte i=0 ; i<20; i++) { 
      digitalWrite(13, HIGH);           // <PROT(13)> : "PanelApp" flashes
      if ( R ) digitalWrite(12, HIGH);  // <INTE(12)> ...when remote
      delay(100);
      digitalWrite(13, LOW);   
      digitalWrite(12, LOW);   
      delay(100);
      if( rst ) break;                  // flash if 1st start 
    }
    
    rst = 1;
    aLed(0);                          
    
// =======================================================================
// ===========================  Main Loop   ==============================
// =======================================================================
  
 do {

// --------------------------- clock: second Timer -----------------------

   if( tick > 0 ) {                                // second timer (clock mode)
      
     if( tRun == 1 ) {                             // --- conter/clock running  ---

       if( R == 0 ) {                              // local mode
         O20 = sw20();                             // read switches A8...3   
         O53 = sw53();
         O86 = sw86();
         if( digitalRead(17) == HIGH ) tick = 0;   //  [A8] off:  stop clock mode
       }

       if( O20 & 1 ) {                             // --- [A0]: display 1/10 sec bar ---   
         if( tPause == 0 ) {
           ls = 0;
           if( O53 & 2 ) {                           // - bar mode: bar -
             if( tick ) {                            // 1...10 -> <stat> 
               for(byte i = 1 ; i < (tick - 1); i++) {
                ls = ls * 2 + 1;
               }
             } 
           }      
           else {                                    // - display mode: dot -  
             if( tick && (tick < 11) ) {                                     
               if( tick > 1 ) ls = 1;
                for(byte i = 1 ; i < (tick - 2); i++) {
                ls = ls * 2;                         // s0...7 (INT...MEMR)
               }
             }  
           }
           sLed(ls);                                 // show
           digitalWrite(13, LOW);
           if( tick > 10 ) digitalWrite(13, HIGH);   // 11 <PROT(13)>(<stat>)
         }
         else digitalWrite(13, LOW);
       }    

           
       if( tick == 1 && tPause == 0 ) {   // execute, not paused... 
         aC++;                            // incr appCounter @ 1s
         if( O86 & 2 ) {                  // clock mode (mm:ss/hh:mm)
           if( (aC % 100) > 59 ) {        // 100:60 calculation 
            aC = aC + 40;                 // 60s -> 1:00
           }
           if( (aC > 3959) ) aC = aC - 3900;   // mm:ss mode: max. 3959
           else if( O86 & 1 ) {                // hh:mm mode: max. 2359
             if( (aC > 2400) ) aC = aC - 2400; // 24:00 -> 00:00                
           }
           txAChar('T');                  // return time
           txAStringLF(String(aC)); 
         }

         sLed(0); cLed(0);
         dClock(O53);                     // --- clock mode display ---             
      
         if( O20 & 2 ) {                  // [A1] Tx led-status
           if( tick == 1 ) txStat();          
         }

         tick = 2;                        // done
       
       }                                  //\ tRun==1
       
       tRun = 2;                          // execution done 
       
     }                                    //\ running     

   }                                      //\ clock mocde

// --------------------------- app timer --------------------------------

     else {                                // app timer
      
       switch(app) {

         case 1:                           // --- counter/timer ---
           if( tC > t80 ) {                // timer counter overflov
             tC = 0; 
             if( tRun && tPause == 0 ) {   // count..., watchdog off
               aC++; digitalWrite(10, LOW);     
             }
             aLed(aC); 
             if( R ) {                               // remote
               if( tick == 0 && O86 & 4 ) tick = 1;  // -> clock mode 
             }
             else {                                  // [A8] -> clock mode
               if( tick == 0 && digitalRead(17) == LOW ) tick = 1; 
             }
           }
           break;

         case 2:                          // --- all random ---
           if( tC > t80 ) {               // timer counter overflow             
             tC = 0;
             byte r = random(4);
             if( R == 0 ) {
               int b = pow(2, random(16));
               la = la ^ b ;               // XOR bit manipulations
               aLed(la);
             } 
             if( R == 1 ) {
               byte b = pow(2, random(8));
               ld = ld ^ b;
               dLed(ld);
             } 
             if( R == 2 ) {
               byte b = pow(2, random(8));
               ls = ls ^ b;
               sLed(ls);
             }
             if( R == 2 ) {
               byte b = pow(2, random(8));
               lc = lc ^ b;
               cLed(lc);
             }  
           }
           break;

         case 3:                          // --- all random block ---
           if( tC > t80 ) {               // timer counter overflow
             tC = 0;
             aLed(random(65553)); 
             dLed(random(255)); 
             sLed(random(255));
             cLed(random(15));
           }
           break;

         case 7:                             // keys & leds test
           aLed(swAdr());                    // adr -> <ADR> 
           dLed(swKey() & 255);              // LSB -> <DATA> 
           sLed(swKey() / 255);              // MSB -> <STATUS>
           digitalWrite(10, millis() & 64);  // watchdog <WAIT(10)>
           break;

         }                                   //\ app    

           
// ---------------------------- CPU Simulation ---------------------------------

         if( (app == 4) && (tC > t80) ) {    // --- CPU Simulation @ timer ----
           tC = 0;
           byte i = 0; 
/*
unsigned int pPC = 0;     // program counter  ++++
unsigned int pA = 0;      // adr pointer
byte pM = 0;              // M-cycle
byte pO = 0;              // actual OP code
byte pR = 0;              // bytes to read
byte pL = 0;              // L-byte
byte pH = 0;              // H-byte
byte pE = 0;              // exec M cycle
*/
             if(pM < 1) pM = 1;           // at least M1

if( digitalRead(66) == LOW ) {
  if( pM != 1 ) { txAString(" +"); txAString(String(pM)); }           
}
             switch(pM) {                 // M-cycle

               case 1:                    // M1: OP code fetch
                 pR = 0; pE = 1;          // reset
                 pO = random(256);        

if( digitalRead(66) == LOW ) {
  txALF();  
  txAString(" PC:"); txAString(String(pPC));
  txAString("  "); txAString(String(pO));
}   
                 sLed(160);               // MEMR, M1
                 aLed(pPC);               // PC -> <A>
                 dLed(pO);                // op code -> <D>
                 i = pO & 15;             // low nibble
                 pPC++;                   
               
                 if( pO < 64 ) {                            // -------- block 00...3f --------
                   if( i == 0 ) {                           // apperance back jumps
                     pPC = pPC - pow(10,(pO/16));    
                     if( pPC > 8192 ) pPC = 1024;
                   }
                   if( (i!=1) && (i!=2) && (i!= 6) && (i!=10) && (i!=14) ) break;  
                   if( i == 1 || i == 14 ) pR = 2;          // 'x1', 'xe' +2 op codes
                   if( i == 2 ) {                           // 'x2' store...
                     pA = random(65535); pE = 10;           // write adr, 1 byte 
                     if( pO == 34 ) {                       // '22' shld NN
                       pR = 2; pM = 2; pE = 9;              // read 2 adr bytes, 2 Bytes to write 
                     }
                     if( pO == 50 )  {                      // '32' sta NN
                       pR = 2; pM = 2; pE = 10;             // read 2 adr bytes, 1 Byte to write
                     }
                                       
                   }
                    if( i == 10 ) {                         // 'xa' load...
                     pA = random(65535); pE = 10;           // store adr, 1 byte 
                     if( pO == 42 || pO == 58 ) {           // '2A', '3A'  lhld NN (lta NN)
                       pR = 2; pM = 2; pE = 13;             // read 2 adr bytes, 1 Bytes to load
                     }               
                   }
                   if( i == 6 ) pR = 1;                     // 'x6' +1 op code
                 }
 
  
                 if( pO > 191 ) {                           // ----- block c0... -----
                   if( random(256) < 16 ) {                 // make rare                
                     if( i && 7 ) {                         // 'x7' & 'xf' restart
                       if( pO == 207 ) pPC = 8;             // 'cf' RST 1                              
                       if( pO == 223 ) pPC = 24;            // 'df' RST 3                              
                       if( pO == 239 ) pPC = 28;            // 'ef' RST 5                              
                       if( pO == 207 ) pPC = 56;            // 'ff' RST 7                              
                       if( pO == 199 ) pPC = 0;             // 'c7' RST 0                              
                       if( pO == 215 ) pPC = 16;            // 'd7' RST 2                              
                       if( pO == 231 ) pPC = 32;            // 'e7' RST 4                              
                       if( pO == 247 ) pPC = 48;            // 'f7' RST 6                                                 
                       pE = 0;
                     }
                       
                     if( i == 1 ) {                         // pops
                       pL = random(256);
                       pH = random(32);
                       pM = 4;                              
                     }                                           
                     if( i == 5 ) {                         // pushs
                       pL = random(256);
                       pH = random(32);
                       pM = 6;                              
                     }                                           

                     if( (i == 6) || (i == 14) ) pR = 1;               // arithm. index

                     if( i == 2 || i == 10 ) {pR = 2; pE = 10; }       // jumps
                     if( pO == 195 || pO == 203 ) {pR = 2; pE = 10; }  
                     if( i == 4 || (i == 12) || (i == 13)) {           // calls
                       pR = 2; pE = 11;
                     }                  
                     if( (i == 8) || (i == 9) ) pE = 12;               // rets

                     if( random(256) < 32 ){                           // make more rare
                       if( pO == 219 ) pR = 1; pE = 14;                // INPUT
                       if( pO == 211 ) pR = 1; pE = 15;                // OUTPUT                   
                     }
                   }
                 }     
                 if( pR ) pM = 2;                           // -> M2   
                 break;

               case 2:                                      // M2:  read L-byte
                 pL = random(256);
                 pA = pL;
                 sLed(128);               // MEMR
                 aLed(pPC);               // PC -> <A>
                 dLed(pL);                // op code -> <D> 
                 if( pR > 1 ) pM = 3;     // 2 bytes -> M3
                 else pM = pE;            // exec. M cycle 
                 pR--;
                 pPC++;                   
                 break;
               
               case 3:                                      // M3:  read H-byte
//                 pH = random(256);
                 pH = random(16);
                 pA = pA + 256 * pH;
                 aLed(pPC);               // PC -> <A>
                 dLed(pH);                // op code -> <D>
                 pM = pE;                 // exec. M cycle
                 pR--;
                 pPC++;
                 break;

               case 4:                                      // POP: read  L-Byte SP
                 sLed(132);                                 // MEMR, STACK
                 aLed(65487);                               // stack pointer 'ffcf'
                 dLed(pL);
                 pM = 5;
                 break;
               case 5:                                      // read  H-Byte SP
                 aLed(65486);                               // stack pointer 'ffce'
//                 dLed(random(256));
                 dLed(pH);
                 pM = 1;                                     // done 
                 break;  

               case 6:                                      // PUSH: write L-Byte SP
                 sLed(134);                                 // MEMR, STACK, WO
                 aLed(65486);                               // stack pointer 'ffcf'
                 dLed(pL);
                 pM = 7;
                 break;
               case 7:                                      // write  H-Byte SP
                 aLed(65487);                               // stack pointer 'ffce'
                 dLed(pH);
                 pM = 1;                                    // done
                 break;  

               case 8:                                      // write 2 bytes 
                 sLed(130);                                  // MEMR, WO
                 aLed(pA - 1);
                 dLed(random(256));
                 pM = 9;
                 break;
               case 9:                                      // write 1 byte 
                 sLed(130);                                 // MEMR, WO
                 aLed(pA);
                 dLed(random(256));
                 pM = 1;
                 break;
                 
               case 10:                                     // M10: jump pHpL -> PC
                 pPC = 255 * pH + pL;
                 pM = 1;
                 break;

               case 11:                                     // M10: call sub pHpL -> PC
                 pL = pPC and 15;
                 pH = pPC / 16;
                 pS = pPC;                                  // save sub adr
                 pPC = 255 * pH + pL;                       // sub adr
                 pM = 6;                                    // PUSH
                 break;

               case 12:                                     // RETURN
                 pPC = pS;                                  // load call adr
                 pL = pPC and 15;
                 pH = pPC / 16;
                 pM = 6;                                    // POP
                 break;

               case 13:                                     // read 1 byte 
                 aLed(pA);
                 dLed(random(256));
                 pM = 1;
                 break;
                     
               case 14:                                     // 219 INPUT  
                 sLed(64);                                  // INP 
                 aLed(random(256));
                 dLed(random(256)); 
                 pM = 1;             
                 break;
                 
               case 15: 
                 sLed(18);                                  // 211 OUTPUT, WO 
                 aLed(random(256));
                 dLed(random(256));                  
                 pM = 1;                               
                 break;   

                          
             }                                     //\ M-cycl.                                                         
         }                                        //\  CPU sim  
                                                   
// ------------------------------------------------------------------------
                 
     }                                       //\ app timer

// ============================  Common  ==================================

// ---------------------- read switches & keys ----------------------------

     if( digitalRead(21) == LOW ) tPause = 1; // [STOP] timer stop
     if( digitalRead(20) == LOW ) tPause = 0; // [RUN] timer run

     if( (lC & 2048) == 2048 ) {             // read switches/Opt -> speed
       if( digitalRead(53) == LOW ) {        // [CLR] 
         aLed(0); dLed(0); sLed(0); cLed(0);
         aC = 0; tC = 0;                     // restart
         pPC = 0;                            // CPU reset
         tRun = 1;                           // start: run
         tick = 1;                           // /s reset
         if ( R ) txAString("     -> [LOCAL]!"); 
         R = 0;                              // remote off
       }
      
       if ( R == 0 ) {                   // local mode                
         t80 = sw80();                   // A0...A9 app timer [876543210]²
         t53 = 8 * (sw53());             // A3...A5 app timer [---543---]²
       }
       else {                             // remote 
         t80 = 64 * O86 + 8 * O53 + O20 ; // option 321 [876543210]
         t53 = 8 * O53;                   // option 2   [---543---]
       }
       t80 = t80 * t80;
       if( t53 == 7 ) t53 = t53 + 2;      // max. (extended)
       t53 = t53 * t53 + 64;              // min. 64

       if( app == 5 ) break;              // exit after analog mode
     }
     
     lC++;                                  // --- loop counter --- 

// ---------------------------  set counter  -------------------------------- 

     if( (lC & 2048) == 2048 ) {                     // --- set counter/clock ---
       if( swKey() & 15 ) {                          // read [AUX] switches 
//         if( O86 > 5 ) tRun = 2;                    // clock: start
         tick = 1;                                   // 1/10 s reset
         digitalWrite(22, LOW);                  
         delay(100);                                 // debounce
         if( ud ) delay(200);                        // hold delay
         if( ud == 1 ) delay(250);                   // 1st hold delay extension
         ud++;
       }
       if( swKey() & 15 ) {                          // still hold...
         byte i = 100;                               // (clock, decimal)
         if( O86 < 6 ) {                             // not clock mode
           if( O53 == 4 ) i = 4096;                  // binary BCD
           if( O53 == 2 ) i = 16;                    // binary bars
         }
         if( digitalRead(30) == LOW ) aC = aC - i;   // AUX1UP    -- decade
         if( digitalRead(31) == LOW ) aC = aC + i;   // AUX1DOWN  ++ decade
         if( digitalRead(32) == LOW ) aC--;          // AUX2UP    -1
         if( digitalRead(33) == LOW ) aC++;          // AUX2DOWN  +1
         if( O86 > 4 ) {                             // clock mode (mm:ss/hh:mm)
           if( (aC % 100) > 59 ) {                   // 100:60 calculation 
            if( digitalRead(32) == LOW ) aC = aC - 40;   // - 60s -> 1:00 
            if( digitalRead(33) == LOW ) aC = aC + 40;   // + 60s -> 1:00            
           }  
           if( O86 & 1 ) {                           // hh:mm mode: max. 1159
             if(aC > 1159) aC = 0;                   // -> 00:00                
           }
           else if(aC > 3959) aC = 0;                // mm:ss mode: max. 3959 -> 0:00
           dClock(sw53());                           // clock -> display ([A5][A4])                                   
         }
         lC = 0;                                     // reset loop counter
         digitalWrite(22, HIGH);                     // ack: <HLDA(22)> on
       }    
       else if( ud ) { ud = 0; digitalWrite(22, LOW); } // setting done 
     }
     
     
     if( tPause == 1 ) {                           // Stop
       if( app == 1 ) {                            // counter/clock
         digitalWrite(10, millis() & 256);         // watchdog <WAIT(10)>
         dClock(sw53());
       }
     }

// ------------------------- clock timer ----------------------------------
       
     if( tick ) {                           // --- clock timer ---
       if( tRun ) {                         // timer flag: running
         if( millis() >= mSec ) {           // --- after 0.1(6) sec ---
           tick++;
           if( tick > 11 ) tick = 1;  
           
           if( O86 ) {                      // sec-mode (all clock modes)
             mSec = millis() + 100;         // next run after 100 ms (1/10 s)
             digitalWrite(10, LOW);         // <WAIT(10)> watchdog off
           }                          
           if( O86 == 7 ) {                 // [A8][A7][A6] min-mode
             mSec = millis() + 6000;        // next run after 6000 ms (60/10 s) 
           }       
           tRun = 1;                        // execute tRun
         }
         if( O53 == 1 ) {                   // --- clock bars mode ---
           m = (aC % 5);                    // 1/5 /min
           if( m ) ma = ( ma | 1024 );      // <A10>
           if( m > 1 ) ma = ( ma | 2048 );  // <A11>
           if( m > 2 ) ma = ( ma | 4096 );  // <A12>
           if( m > 3 ) ma = ( ma | 8192 );  // <A13>
           aLed(ma);                              
           
           if( O20 & 4 ) {                  // - marker option -
             if ( (millis() & 15) == 7 ) {  // 1/1000s @3:   mark clock bar bits
               aLed(15423);                 // - -** **- --- *** ***             
               dLed(63);                    //            -- *** ***  
               if( O20 & 2 ) delay(1);      // bright marker option
               aLed(ma); dLed(md);          // restore leds 
             }     
           }
         } 
         
         if( O53 == 4) {                    // --- BCD-mode ---
           if( O20 & 6 ) {                  // BCD marker option
             if ( (millis() & 15) == 7 ) {  // 1/1000s @3:   mark BCD bits
               ma = la; md = ld;
               if( O86 & 3 ) aLed(463);     // clock mode    - --- --- *** --* *** 
               else aLed(62415);            // counter mode  * *** --* *** --* ***
               dLed(207);                   //                          ** --* ***
               if( O20 & 2 ) delay(1);      // bright marker option
               aLed(ma); dLed(md);          // restore leds 
             }     
           }
         }  
 
         else if( O86 & 1 ) {                                  // --- [A6]: min-mode --- 
           if( tPause == 0) digitalWrite(12, millis() & 1024); // INTE(12) blink
           if( R == 0 ) {
             O86 = sw86(); O53 = sw53();                       // read switches
           }
         }
         
       }                                      //\ running  
     }                                        //\ clock mode
     else tC++;                               //\ counter mode: timer counter   
   
     if( rxSerial() == 'X') break;            // ASCII 'X'  -> exit AppMode 
                              
   }                                          //\ do
   while( digitalRead(52) == HIGH );          //  key [RESET] -> Exit 

// =====================================================================      
// ========================== LOOP until STOP ==========================
// =====================================================================                                   

    aLed(0); dLed(0); sLed(0); cLed(0);        // --- EXIT ---
  
    txAString("     Exit: ");
    txAHex(aRx); 
    txALF();
    
    closeAPort();

    if( aRx ) digitalWrite(13, HIGH);         // <PROT(13)> return value for cmd exit
    delay(500);
    
}    

// =============================== EXIT ================================
  
