// -----------------------------------------------------------------------------
// PanelMode for David Hensel's Altair 8800 Simulator  
// Copyright (c) 2021 Ralf Lampe www.erel.de/Projekte/Altair
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
/*
===============================================================
Command list Version 1:
===============================================================
Data format: 19200-8-N-1
             commands as upper case or signs utf8 characters
             nibbles  as low case Hex-characters "0"..."f"
             without or with linefeeds (subpressed) 
              
Commands:   Port -> ALTAIR-DUINO
---------------------------------------------------------------
Aaaaa       Set Adr-LEDs (+4 Nibles)
Ddd         Set Data-LEDs (+2)
Ssss        Set Status-LEDs (+3)
Laaaaddsss  Set all LEDs (+9) 
+   Set all LEDs
-   Clear all LEDs
#   Poll switches and keys (STOP...AUX)
*   KeyChange auto tx on
R   Reset (C/E, W, * off) 
C   Command
E   Echo on       
W   Watchdogs (flashing: WAIT(10), HLDA(22) when cmd pendig)
Q   Quit Panel Mode 


Messages:    ALTAIR-Duino -> Port
--------------------------------------------------------------
After "Reset" returns:
Command character*

"Echo on" returns:
echo rx characters until command detected:

"Command" returns:
result  ([LF])dd[LF]          2 nibbles   
        ([LF])sss[LF]         3
        ([LF])aaaa[LF]        4
        ([LF])ttttkkkk[LF]    8
        ([LF])aaaaddsss[LF]*  9

*when "auto key change" then also returns, if toggles/keys changed


Characters returned:
dd    2 nibbles data LED
sss   3 nibbles status LED
aaaa  4 nibbles address LED
tttt  4 nibbles toggle switch (Rx)
ssss  4 nibbles key           (Rx)
===============================================================
*/
// -----------------------------------------------------------------------------

#include "PanelMode.h"
#include "PanelApp.h"

static byte pPort = 0;     // COM-port, default 0 (USB program)
byte rx = 0;               // rx character
byte rb[12];               // rx buffer                   
byte rn = 0;               // rx nibble counter
byte ca = 0;               // actual command
byte cn = 0;               // nibbles/command
byte n = 0;                // nibble value 
byte nc = 0;               // nibble counter         
byte ss[10];               // switch status array
byte sc = 0;               // ...changed
byte st = 0;               // ...tx flag 
byte el = 0;               // LF flag for switch state  
byte ef = 0;               // valid rx echo flag
byte i = 0;               
unsigned int l = 0;        // loop counter
byte wd = 1;               // INTE(12)-LED as watchdog  


// -------------- Funktionen ---------------

void txChar(byte tc) {               //tx character to port tc
    switch (pPort) {                              
        case 0: Serial.write(tc); break;
        case 1: Serial1.write(tc); break;
        case 2: Serial2.write(tc); break;
        case 3: Serial3.write(tc); break;
       }
}
void txHex(byte th) {                //tx hex value
    switch (pPort) {                             
        case 0: Serial.print(th, HEX); break;
        case 1: Serial1.print(th, HEX); break;
        case 2: Serial2.print(th, HEX); break;
        case 3: Serial3.print(th, HEX); break;
       }
}
void txLF() {                       //tx linefeed
    switch (pPort) {                             
        case 0: Serial.println(); break;
        case 1: Serial1.println(); break;
        case 2: Serial2.println(); break;
        case 3: Serial3.println(); break;
       }
}

void txString(String tx) {          //tx string 
    switch (pPort) {                           
        case 0: Serial.print(tx); break;
        case 1: Serial1.print(tx); break;
        case 2: Serial2.print(tx); break;
        case 3: Serial3.print(tx); break;
    } 
}

void txStringLF(String tx) {          //tx string 
    switch (pPort) {                           
        case 0: Serial.print(tx); break;
        case 1: Serial1.print(tx); break;
        case 2: Serial2.print(tx); break;
        case 3: Serial3.print(tx); break;
    }
    txLF();   
}

void txStart() {                      // tx start message to configured port
    txLF(); txLF(); txLF;                      
    txStringLF(" --- PanelMode (c) 2021 Ralf Lampe   www.EREL.de --- ");
    txStringLF("     Commands: D(2), A(4), S(3), L(9), +, -, #, *");
    txStringLF("     Control:  T(4), P(4), I, C, E, Q");
    txStringLF("               (): all Parameters are hex-nibbles");      
}

byte ascval(byte a) {              // conversion ASC/Hex (0...9a...f) -> value 
  byte z = a - 48;                 // "0"..."9","a"..."f" = %48...&57,%97...%102     
  if( z > 48 ) z = z - 39;         // 0...9, ?"a"..."f" = %49...%54 -> 10...15
  return z;                         
}

byte findcmd(byte z) {             // look for parametric commands
  switch(z) {
    case 'D': cn = 2; return z;
    case 'A': cn = 4; return z;
    case 'P': cn = 4; return z;
    case 'T': cn = 4; return z;    // set time
    case 'S': cn = 3; return z;
    case 'L': cn = 9; return z;
  }
  return 0; 
}


void aLED() {                      // set adress leds (nibble)
  n = rb[nc];                      //1. Nibble -> Pin 34...37
  digitalWrite(47, (n & 1));
  digitalWrite(46, (n & 2));
  digitalWrite(45, (n & 4));
  digitalWrite(44, (n & 8));  
  nc++; 
  n = rb[nc];                      //2. Nibble -> Pin 38...41
  digitalWrite(51, (n & 1));   
  digitalWrite(50, (n & 2));
  digitalWrite(49, (n & 4));
  digitalWrite(48, (n & 8));
  nc++;
  n = rb[nc];                      //3. Nibble -> Pins 51,50,49,48
  digitalWrite(38, (n & 1));
  digitalWrite(39, (n & 2));
  digitalWrite(40, (n & 4));
  digitalWrite(41, (n & 8));
  nc++;
  n = rb[nc];                      //4. Nibble -> Pin 47...44
  digitalWrite(34, (n & 1));
  digitalWrite(35, (n & 2));
  digitalWrite(36, (n & 4));
  digitalWrite(37, (n & 8));
  nc++;
}

void wLED(unsigned int led) {        // set address leds (word) 
  digitalWrite(34, (led & 1));       // A0...A3 -> Pin 34...37
  digitalWrite(35, (led & 2));
  digitalWrite(36, (led & 4));
  digitalWrite(37, (led & 8));  
  digitalWrite(38, (led & 16));      // A4...A7 -> Pins 38...41   
  digitalWrite(39, (led & 32));
  digitalWrite(40, (led & 64));
  digitalWrite(41, (led & 128));
  digitalWrite(51, (led & 256));     // A8...A11 -> Pin 51...48  
  digitalWrite(50, (led & 512));
  digitalWrite(49, (led & 1024));
  digitalWrite(48, (led & 2048));
  digitalWrite(47, (led & 4096));    // A12...A15 -> Pin 47...44 
  digitalWrite(46, (led & 8192));
  digitalWrite(45, (led & 16384));
  digitalWrite(44, (led & 32768)); 
}

void dLED() {                      // set data leds
  n = rb[nc];                      //1. Nibble -> Pin 25...28
  digitalWrite(14, (n & 1));
  digitalWrite(15, (n & 2));
  digitalWrite(29, (n & 4));
  digitalWrite(11, (n & 8));
  nc++; 
  n = rb[nc];                      //2. Nibble -> Pins 14,15,29,11
  digitalWrite(25, (n & 1));
  digitalWrite(26, (n & 2));
  digitalWrite(27, (n & 4));
  digitalWrite(28, (n & 8));
  nc++; 
}


void sLED() {                      // set status leds
  n = rb[nc];                      // 1. Nibble PROT(13),INTE(12),WAIT(10),HLDA(22) Pin 13,12,10,22
  digitalWrite(13, (n & 1));       // PROT(13) 
  digitalWrite(12, (n & 2));       // INTE(12)
  digitalWrite(10, (n & 4));       // WAIT(10)
  digitalWrite(22, (n & 8));       // HLDA(22)   
  nc++;
  n = rb[nc];                      // 2. Nibble OUT,M1,INP,MEMR Pin 6...9
  digitalWrite(6, (n & 1));
  digitalWrite(7, (n & 2));
  digitalWrite(8, (n & 4));
  digitalWrite(9, (n & 8));
  nc++;
  n = rb[nc];                      // 3. Nibble INT,WO,STACK,HLTA Pin 2...5
  digitalWrite(2, (n & 1));
  digitalWrite(3, (n & 2));
  digitalWrite(4, (n & 4));
  digitalWrite(5, (n & 8));
  nc++; 
}

void rSwitches() {                 // get switch values
  n = 8;                           // 8. Nibble AUXU,...D,AUX2U,...D Pin 30...33 
  i = 0;                            
  if( digitalRead(30) == LOW ) i = i + 8;        
  if( digitalRead(31) == LOW ) i = i + 4;        
  if( digitalRead(32) == LOW ) i = i + 2;          // read -> statusnibble
  if( digitalRead(33) == LOW ) i = i + 1;        
  if( i != ss[n] ) sc = 1;         
  ss[n] = i;                       

  n = 7;                           // 7. Nibble RES,CLR,PROT(13),UNPR Pin 52,53,60,61 
  i = 0;                           // reset statusnibble 
  if( digitalRead(52) == LOW ) i = i + 8;        
  if( digitalRead(53) == LOW ) i = i + 4;        
  if( digitalRead(60) == LOW ) i = i + 2;          // read -> statusnibble
  if( digitalRead(61) == LOW ) i = i + 1;        
  if( i != ss[n] ) sc = 1;         // status changed -> flag
  ss[n] = i;                       // ...save

  n = 6;                           // 6. Nibble EXAM,NEXT,DEPO,NEXT  Pin  56...59
  i = 0;                           
  if( digitalRead(56) == LOW ) i = i + 8;          // read -> statusnibble
  if( digitalRead(57) == LOW ) i = i + 4;        
  if( digitalRead(58) == LOW ) i = i + 2;        
  if( digitalRead(59) == LOW ) i = i + 1;        
  if( i != ss[n] ) sc = 1;         
  ss[n] = i;                      

  n = 5;                           // 5. Nibble STOP,RUN,STEP,SLOW Pin 21,20,54,55 
  i = 0;                       
  if( digitalRead(21) == LOW ) i = i + 8;        
  if( digitalRead(20) == LOW ) i = i + 4;          // read -> statusnibble
  if( digitalRead(54) == LOW ) i = i + 2;        
  if( digitalRead(55) == LOW ) i = i + 1;        
  if( i != ss[n] ) sc = 1;        
  ss[n] = i;                     

  // -----------

  n = 4;                           // 4. Nibble S0...3 Pin 62...65 (A0...A3) 
  i = 0;                           
  if( digitalRead(62) == LOW ) i = i + 1;          // read -> statusnibble
  if( digitalRead(63) == LOW ) i = i + 2;        
  if( digitalRead(64) == LOW ) i = i + 4;        
  if( digitalRead(65) == LOW ) i = i + 8;        
  if( i != ss[n] ) sc = 1;        
  ss[n] = i;                      

  n = 3;                           // 3. Nibble S4...7 Pin 66...69 (A4...A7) 
  i = 0;                         
  if( digitalRead(66) == LOW ) i = i + 1;          // read -> statusnibble
  if( digitalRead(67) == LOW ) i = i + 2;        
  if( digitalRead(68) == LOW ) i = i + 4;        
  if( digitalRead(69) == LOW ) i = i + 8;        
  if( i != ss[n] ) sc = 1;       
  ss[n] = i;                    

  n = 2;                           // 2. Nibble S8...11 Pin 17,16,23,24 (A8...A11) 
  i = 0;                          
  if( digitalRead(17) == LOW ) i = i + 1;          // read -> statusnibble
  if( digitalRead(16) == LOW ) i = i + 2;        
  if( digitalRead(23) == LOW ) i = i + 4;        
  if( digitalRead(24) == LOW ) i = i + 8;        
  if( i != ss[n] ) sc = 1;       
  ss[n] = i;                       

  n = 1;                           // 1. Nibble S12...15 Pin 70,71,42,43 (A12...A15) 
  i = 0;                          
  if( digitalRead(70) == LOW ) i = i + 1;          // read -> statusnibble
  if( digitalRead(71) == LOW ) i = i + 2;        
  if( digitalRead(42) == LOW ) i = i + 4;        
  if( digitalRead(43) == LOW ) i = i + 8;        
  if( i != ss[n] ) sc = 1;
  ss[n] = i;             
}

void res() {                      // reset
  rx = 0;   
  rn = 0;
  nc = 0;     
  cn = 0;
  ca = 0;
  el = 0;
}

void ok() {                       // ok aknowledge                                   
  i = 0;
  if( ef == 2 ) txLF();
  if( ef > 0 ) {                  // echo number of data nibbles
    switch (ca) {                             
       case 'D': cn = 2; break;
       case 'S': cn = 3; break;
       case 'A': cn = 4; break;
       case 'P': cn = 4; break;
       case 'T': cn = 4; break;
       case 'L': cn = 9; break;
    }
    for (byte i = 1; i < nc ; i++) { // ...data
      txHex(rb[i]);                
    }
    txLF();
  }
  else txChar(ca);                 // ACK command
  res();
}

void ack() {                       // ACK rx command                        
  if( ef > 1 ) txLF();             // echo
  txChar(rx);                      // ... command
  if( ef > 0 ) txLF();             // echo 
  res();
}


void confPins() {                                         // config arduino pins 
//  for (byte i = 2; i < 72; i++) pinMode(i, INPUT);        // first all IN
  pinMode(20, INPUT);                                     // D20 [RUN]  INPUT
  for (byte i = 2; i < 16; i++) pinMode(i, OUTPUT);       // D2...15  OUT
  digitalWrite(16, HIGH); digitalWrite(17, HIGH);         // D16, 17  PULL UP   
  pinMode(22, OUTPUT);                                    // D18      OUT 
  for (byte i = 19; i < 22; i++) digitalWrite(i, HIGH);   // D19...21 PULL UP 
  pinMode(22, OUTPUT);                                    // D22      OUT 
  digitalWrite(23, HIGH); digitalWrite(24, HIGH);         // D23, 24  PULL UP  
  for (byte i = 25; i < 30; i++) pinMode(i, OUTPUT);      // D25...29 OUT
  for (byte i = 30; i < 34; i++) digitalWrite(i, HIGH);   // D30...33 PULL UP
  for (byte i = 34; i < 42; i++) pinMode(i, OUTPUT);      // D34...41 OUT
  digitalWrite(42, HIGH); digitalWrite(43, HIGH);         // D42, 43  PULL UP 
  for (byte i = 44; i < 52; i++) pinMode(i, OUTPUT);      // D44...51 OUT
  for (byte i = 52; i < 72; i++) digitalWrite(i, HIGH);   // D52...71 PULL UP
}

void openPorts() {  
  switch(pPort) {                      // open user port for input        
//      case 0: Serial.begin(19200); delay(200); Serial.println("   USB remote"); 
      case 0: Serial.begin(19200); break;
//     case 1: Serial1.begin(19200); delay(200); Serial1.println("   Serial 1 remote"); 
      case 1: Serial1.begin(19200); break;
      case 2: Serial2.begin(19200); break; 
      case 3: Serial3.begin(19200); break;
  }   
}    
void closePorts() {  
  switch(pPort) {                      // close user port        
      case 0: Serial.end(); break; 
      case 1: Serial1.end(); break;
      case 2: Serial2.end(); break;
      case 3: Serial3.end(); break;
  }   
}    
// ---------------------------------------------
// --------------- Main programm ---------------
// ---------------------------------------------


PanelMode::PanelMode(byte pin) {

/*
  Serial.begin(19200);  delay(200); Serial.println(" Panel Mode USB "); delay(200); Serial.end();
  Serial1.begin(19200); delay(200); Serial1.println(" Panel Mode Serial 1 ");delay(200); Serial1.end();
  Serial2.begin(19200); delay(200); Serial2.println(" Panel Mode Serial 2 ");delay(200); Serial2.end();
  Serial3.begin(19200); delay(200); Serial3.println(" Panel Mode Serial 3 ");delay(200); Serial3.end();
*/
    confPins();                                // Set pins as in- or outputs
    
    // --- read config ports ----   

    pPort = 0;                                 // Reset, switch pins L-active
    if( digitalRead(70) == LOW ) pPort = 3;    // A12: Tx 19,18 -DUE- (ESP...)
    if( digitalRead(71) == LOW ) pPort = 2;    // A13: Tx 17,16 -DUE-
    if( digitalRead(42) == LOW ) pPort = 1;    // A14: Tx 15,14 -DUE-

    Serial.begin(19200);  delay(200); Serial.print(" Remote Port: "); Serial.println(pPort); delay(200); Serial.end();
    Serial1.begin(19200); delay(200); Serial1.print(" Remote Port: "); Serial1.println(pPort);delay(200); Serial1.end();
    Serial2.begin(19200); delay(200); Serial2.print(" Remote Port: "); Serial2.println(pPort);delay(200); Serial2.end();
    Serial3.begin(19200); delay(200); Serial3.print(" Remote Port: "); Serial3.println(pPort);delay(200); Serial3.end();

    Serial.begin(115200);                       // Start message to USB program port
    Serial.print("Starting PanelMode! Please switch to Port ");
    Serial.print(pPort);
    Serial.print(" at 19200 Baud...");
    Serial.end();

    openPorts();                             // open remote com port
    delay(500);
    txLF(); txLF(); 
                
    do {                                       // checking for PanelApps
      i = 0;                              
      if( digitalRead(16) == LOW ) i = 1;      // A9
      if( digitalRead(23) == LOW ) i = 1;      // A10
      if( digitalRead(24) == LOW ) i = 1;      // A11
      if( i > 0 ) {
        closePorts();
        PanelApp(pPort, 0, 0, 0, 0);           // --- call Panel-App (Port, #) ---     
        openPorts();
      }
      if( digitalRead(13) == HIGH ) break;     // <PROT(13)>: exit after 'X'-cmd 
      if( digitalRead(43) == HIGH ) i = 0;     // A15 released: exit PanelMode
    }
    while (i > 0);                             // Loop until A9...A11 & A15 off 

    openPorts();                               // open remote com port

    if( pPort == 1 ) {                         // Wifi module: "waiting for network"
            txStringLF(" >>> Wait 60 sec for network connection...");
            for (byte i = 0; i < 16; i++) {            // 16 * 
              wLED(pow(2, i));                         // time-left dots
              txString("    waiting... (cancel: [RUN]) ");                                 
              txStringLF(String(15 - i));
              for (n = 1; n < 75; n++) {               //      75 *  
                delay(50);                             //          0.05 sec = 60 sec
                digitalWrite(10, n % 2);               // <WAIT(10)> flashing
                digitalWrite(12, n % 2);               // <INTE(12)>
                if (digitalRead(20) == LOW) {          // [RUN] breaks waitung
                  i = 16; 
                  n = 75; 
                }
              }
            }
            wLED(0);
            digitalWrite(10, LOW);                     // <WAIT(10)> flashing
            digitalWrite(12, LOW);                     // <INTE(12)> 
            n = 0;
    }
             
    delay(500);
    txStart();                                 // tx start message to configured port
                    
    for(byte i=0 ; i<20; i++)             
    { 
     digitalWrite((44 + pPort), LOW);          // LED A15...A12 (Pin 44...47)
     delay(100);                   
     digitalWrite((44 + pPort), HIGH);         // flashes 
     delay(100);
    }          

    digitalWrite(12, HIGH);                    // <INTE(12)>: "PanelMode"
    wd = 1;                                    // ...as watchdog

// =======================================================================
// ===========================  Main Loop   ==============================
// =======================================================================
    
   do {

      l++;                                          // loop counter (timer) 
      rx = 0;                                       // CLR rx
      
      switch (pPort) {                      // check for received byte on config. port        
        case 0: if( Serial.available() ) rx = Serial.read(); break;
        case 1: if( Serial1.available() ) rx = Serial1.read(); break;
        case 2: if( Serial2.available() ) rx = Serial2.read(); break;
        case 3: if( Serial3.available() ) rx = Serial3.read(); break;
      }

      if( rx > 31 )                  // valid utf8 character... 
      {  
        if( findcmd(rx) > 0 ) {      // check for command {'D','A','S','L','P','T')
          ca = rx;                   // set as actual command  
          rn = 0;                    // Reset rx counter (cn already set in findcmd)                                     
        }       
        else                         // check if data byte 
        {
          if( ca > 0 ) {             // after active command received 
            n = ascval(rx);          // conversion ASC/Hex (0...9a...f) -> nibble 
            if( n < 16 ) {                  
              if( rn < 10 ) rn++;    // incr rx nibble counter, max. 10        
              rb[rn] = n;            // save to nibble buffer
            }
          }     
        }
               
        if( ef == 2 ) txChar(rx);          // echo valid Rx charcter
        nc = 1;                            // Reset nibble counter                 
        
        if( rn == cn ) {                   // expected number of nibbles/command 
          switch (ca) {                    // look for parametric cmds ('D','A','S','L','P','T'}        
            case 'P':                      // remote start PanelApp...
              ok();                        // aknowledge
              closePorts();
              PanelApp(pPort, rb[1], rb[2], rb[3], rb[4]);   // --- call Panel-App (Port, #) ---     
              openPorts();
              txLF(); txStringLF(" --- Resume PanelMode ---");
              digitalWrite(12, HIGH);      // <INTE(12)>      
              break;
            
            case 'T':                      // set time PanelApp...
              ok();                        // aknowledge
              PanelApp(pPort, (rb[1] + 8), rb[2], rb[3], rb[4]);  // --- call Panel-App (Port, #) ---     
              openPorts();
              txLF(); txStringLF(" --- Time set done ---");
              break;
            
            case 'D': dLED(); ok(); break; // 2 nibbles "dd" -> data leds 
            case 'S': sLED(); ok(); break; // 3 nibbles "sss" -> status leds 
            case 'A': aLED(); ok(); break; // 4 nibbles "aaaa" -> address leds 
            case 'L':
              if( rn == 9 ) {              // 9 nibbles "ddaaaabbs" -> leds 
                dLED();                    // -> data
                aLED();                    // -> adr
                sLED();                    // -> staus
                ok();                      // ready, ack.
              }
              break; 
          }                              //\ sw. param cmds
        }                                //\ exp. nibbles                             
        
        switch (rx) {                    // look for control commands...
          case '+':                                   // all leds on
            for (byte i = 1; i < 10; i++) rb[i] = 15; // all nibbles = "f" 
            rn = 9;
            aLED();
            dLED();
            sLED(); 
            if( ef > 0 ) ok();
            else ack();
            break;
          case '-':                                   // all leds off
            for (byte i = 1; i < 10; i++) rb[i] = 0;  // all nibbles = 0 
            rn = 9;
            aLED();
            dLED();
            sLED(); 
            if( ef > 0 ) ok();
            else ack();
            break;   
          case '#':
            rSwitches();                               // read all switch status -> ss[]  
            if( ef == 0 ) txChar(rx); 
            if( ef != 1 ) txLF();                     
            for (byte i = 1; i < 9; i++) txHex(ss[i]); // tx statusarray 
            txLF();                                    // echo
            ca = 0;                                    // reset
            break;
          case '*':                                    // auto tx when switches change on
            st = 1;
            ack();                                     // aknowledge
            break;
          case '?':                                    // alive?
            ack();
            txStringLF("!");
            break;  
          case 'I':                                    // INITIALIZE
            st = 0;                                    // clear switch tx flag 
            ef = 0;                                    // clear echo flag
            wd = 0;                                    // watchdog off
            ack();                                     // aknowledge
            txStringLF("     Reset done");
            break;
          case 'R':                                    // "RESET" (WiFi error)
            digitalWrite(10, HIGH);                    // <WAIT(10)> flashes
            delay(100);
            digitalWrite(10, LOW);
            delay(100);
            digitalWrite(10, HIGH);
            break;
          case 'C':                                    // command ack mode
            ef = 1;                                    // mode: 1
            ack();
            break;
          case 'E':                                    // echo mode
            ef = 2;                                    // mode: 2 (echo rx) & (cmd ack) 
            ack();                    
            break; 
          case 'W':                                    // watchdog on
            wd = 1;                                    // <WAIT(10)> flashing
            ack();                                     // aknowledge
            break;
                      
          }                        //\ sw. non par cmds

        }                          //\ valid utf8 character... 
        
        else {                                         // ---- iddle ----*
          if( st == 1 ) {                              // auto switch flag active
            if( l > 32768 ) {                          // read cycle
              if( wd > 0 ) {
                if( digitalRead(12) == HIGH ) digitalWrite(12, LOW); // watchdog: <INTE(12)> flashes     
                else digitalWrite(12, HIGH);                          
                if( ca > 0 ) digitalWrite(13, digitalRead(12));      // also <PROT(13)> when cmd pending
                else digitalWrite(13, LOW);              
              }
              sc = 0 ;                                     // Reset  
              l = 0;
              rSwitches();                                 // get switch status 
              if( sc > 0 ) {                               // changed...
                if( el == 0 ) { txLF(); el = 1; }          // echo line feeded flag, set 
                for (byte i = 1; i < 9; i++) txHex(ss[i]); // tx switch status array 
                txLF();
                delay(100);                                // debounce                                             
              }            
            }
          }   
          else { 
            if( wd > 0 ) {                                // watchdog: 
//              n = l / 64; 
              digitalWrite(12, l & 4096);                 // <INTE(12)> flashes
              if( cn > 0 ) digitalWrite(13, (l & 8192));  // <PROT(13)> flashes when cmd pending
              else digitalWrite(13, LOW);
            }
          }
        }                                                 //\ iddle
      }                                                   //\ do      
      while (rx != 'Q');  // ---------- LOOP until "Q" /quit) --------------------------

      for (byte i = 1; i < 10; i++) rb[i] = 0;   // close PanelMode 
        rn = 9;                                  // LEDs off  
        aLED(); dLED(); sLED();  
      digitalWrite(13, HIGH);                    // <INTE(12)> pin 10
                                      
      txStringLF("Stop PanelMode");
      txStringLF("Try to start ALTAIR-Simulator at 115200 Baud...");
      txStringLF("Maybe, it's necessary to repower with A15 switched off");
 
      closePorts();                              // release all ports         

      delay(500);

      Serial.begin(115200);               // Message to USB program port
      Serial.println("Now starting ALTAIR-Simulator...");
      Serial.end();

} 

  
